#ifndef _DynamicColorScaleCurator_h__
#define _DynamicColorScaleCurator_h__

#include "Client/ClientUtils/Network/NetworkInterface.h"
#include <buildspec.h>

#include <boost/thread/recursive_mutex.hpp>

namespace GST
{
namespace ClientUtils
{
/**
 * Use this class as a DynamicColorScale container and fetcher.
 *
 * Not every scale need to be loaded complete at runtime. The main purpose of
 * this class is to encapsulate the code of fetching. The DynamicColorValues are
 * required just for matched scale. If a scale matches call fetchByPosition() or
 * fetchById() to load the color values. If the scale is required again it is
 * not required to load the values from the database again because they are
 * fetched already.
 *
 * This class is thread safe.
 *
/code
  DynamicColorScaleCurator myCurator;
  myCurator.setNetwork(network);
  myCurator.initialize();
  DynamicColorScaleListPtr scales = myCurator.getScales();

  int pos=0;
  for(DynamicColorScalePtr scale : *scales)
  {
	 if( scale->MatchByName(someAttributeName) )
	 {
		myCurator.fetchById(scale->getId());
		break;
	 }
	 }
	 ++pos;
  }

  scale.at(i)->GetColorValue(someAttributeValue);
/endcode
 *
 */
class GST_API_EXPORT DynamicColorScaleCurator
{
public:
	/**
	 * Sets the NetworkPtr the currator is working on. This pointer can be
	 * change while lifetime of DynamicColorScaleCurator
	 */
	void setNetwork(NetworkPtr network);

	/**
	 * Initializes intern list of scales by calling
	 * NetworkInterface::ListDynamicColorScales()
	 */
	void initialize();

	/**
	 * Returns the list of DynamicColorScales. Items in list can be emtpy (need
	 * to fetch) or be fetched already. Call fetchById() or fetchByPosition()
	 * with the items of this list which are DynamicColorScale::empty() in order
	 * to load the values.
	 */
	DynamicColorScaleListPtr getScales() const;

	/**
	 * Forwards to network method.
	 * NetworkInterface::CreateUpdateDynamicColorScale() and update the internal
	 * scales field.
	 */
	void CreateUpdateDynamicColorScale(DynamicColorScalePtr dyncs);

	/**
	 * Forwards to network method.
	 * NetworkInterface::DropDynamicColorScale() and update the internal
	 * scales field.
	 */
	void DropDynamicColorScale(const DynamicColorScale::ID &dyncs_id);

	/**
	 * Fetches/Load DynamicColorValues from the database for the specified
	 * scale. The scaleIdx is the position in the intern list returned by
	 * getScales()
	 */
	void fetchByPosition(int scaleIdx);

	bool fetchRequired(int scaleIdx) const;

	/**
	 * Fetches/Load DynamicColorValues from the database for the specified
	 * scale. The scaleIdx is the DynamicColorScale::id()
	 */
	void fetchById(DynamicColorScale::ID id);

	DynamicColorScalePtr getById(DynamicColorScale::ID id,
								 int *position = NULL) const;

protected:
	NetworkPtr m_network;
	DynamicColorScaleListPtr m_scales;
	/**
	 * Supervising resources
	 *  -# m_nework
	 *  -# m_scales
	 */
	mutable boost::recursive_mutex m_access;
};

} // namespace ClientUtils
} // namespace GST

#endif // _DynamicColorScaleCurator_h__
